'use client';

import { useState, useEffect, useRef } from 'react';
import { createPortal } from 'react-dom';
import Image from 'next/image';
import Link from 'next/link';
import { useTranslations, useLocale } from 'next-intl';
import { useRouter, usePathname } from 'next/navigation';
import { locales } from '@/config';

export default function LakeTokenHeader() {
  const [isMenuOpen, setIsMenuOpen] = useState(false);
  const [isLanguageOpen, setIsLanguageOpen] = useState(false);
  const t = useTranslations('LakeToken');
  const locale = useLocale();
  const router = useRouter();
  const pathname = usePathname();
  const langMenuRef = useRef<HTMLDivElement>(null);

  useEffect(() => {
    if (!isLanguageOpen) return;
    function handleClickOutside(event: MouseEvent) {
      if (langMenuRef.current && !langMenuRef.current.contains(event.target as Node)) {
        setIsLanguageOpen(false);
      }
    }
    document.addEventListener('mousedown', handleClickOutside);
    return () => document.removeEventListener('mousedown', handleClickOutside);
  }, [isLanguageOpen]);

  useEffect(() => {
    const handleScroll = () => {
      if (isLanguageOpen) {
        setIsLanguageOpen(false);
      }
    };

    window.addEventListener('scroll', handleScroll);
    return () => window.removeEventListener('scroll', handleScroll);
  }, [isLanguageOpen]);

  const handleLocaleChange = (newLocale: string) => {
    const currentPathname = window.location.pathname;
    const newPath = `/${newLocale}${currentPathname.startsWith(`/${locale}`) ? currentPathname.substring(locale.length + 1) : currentPathname}`;
    router.replace(newPath);
    setIsLanguageOpen(false);
  };

  const toggleMenu = () => {
    setIsMenuOpen(!isMenuOpen);
  };

  const toggleLanguageMenu = () => {
    setIsLanguageOpen((v) => !v);
  };

  const menuItems = [
    { key: 'main_page', weight: 'font-normal', path: '/' },
    { key: 'lake_token', weight: 'font-bold' },
    { key: 'why_lake_stands', weight: 'font-normal' },
    { key: 'safety_header', weight: 'font-normal' },
    { key: 'roadmap_highlights', weight: 'font-normal' },
    { key: 'tokenomics', weight: 'font-normal' },
    { key: 'buy_lake_token', weight: 'font-normal' },
    { key: 'contact_us', weight: 'font-normal' }
  ];

  // Определяем куда должна вести стрелка в зависимости от текущей страницы
  const getBackArrowHref = () => {
    if (pathname.includes('/lake-token/security')) {
      return `/${locale}/lake-token`; // На странице Security -> ведет на LakeToken
    } else {
      return `/${locale}`; // На странице LakeToken -> ведет на главную
    }
  };

  return (
    <header className="w-full bg-[#EFEFEF] relative z-[99999]">
      {/* Стрелочка назад */}
      <Link
        href={getBackArrowHref()}
        className="lake-token-back-arrow"
        aria-label="Назад"
        style={{position: 'absolute', left: 20, top: '50%', transform: 'translateY(-50%)', zIndex: 100000, padding: 0, display: 'flex', alignItems: 'center', justifyContent: 'center', width: 36, height: 36}}
      >
        <svg width="28" height="28" viewBox="0 0 28 28" fill="none" xmlns="http://www.w3.org/2000/svg">
          <path d="M18 6L10 14L18 22" stroke="#002D64" strokeWidth="2.5" strokeLinecap="round" strokeLinejoin="round"/>
        </svg>
      </Link>
      {/* Основной хедер */}
      <div className="w-full py-5">
        <div className="lake-token-container flex items-center">
          {/* Логотип */}
          <div className="w-[152px] h-[22px] relative">
            <Image
              src="/images/lake-token/header/LAKE Token.svg"
              alt="LAKE Token"
              fill
              sizes="152px"
              className="object-contain"
            />
          </div>
          {/* Правая часть */}
          <div className="flex items-center gap-4 ml-auto">
            {/* Языковой переключатель */}
            <div className="relative" ref={langMenuRef}>
              <button
                className="flex items-center gap-2 cursor-pointer"
                aria-label="Toggle language menu"
                tabIndex={0}
                onClick={toggleLanguageMenu}
              >
                <div className="w-4 h-4 relative">
                  <Image
                    src="/images/lake-token/header/Mask group.png"
                    alt="Globe"
                    fill
                    sizes="16px"
                    className="object-contain"
                  />
                </div>
                <span className="text-[#002D64] text-lg font-medium font-inter">|</span>
                <span className="text-[#002D64] text-lg font-medium font-inter uppercase">
                  {locale === 'ru' ? 'РУС' : locale === 'en' ? 'ENG' : locale === 'ar' ? 'العربية' : locale === 'zh' ? '中文' : locale.toUpperCase()}
                </span>
              </button>
              {isLanguageOpen && typeof window !== 'undefined' && createPortal(
                <div
                  ref={langMenuRef}
                  className="absolute right-0 mt-2 w-24 bg-[#EFEFEF] rounded-lg shadow-lg py-1 z-[100000] animate-slideDownFadeInMenu"
                  style={{ top: langMenuRef.current?.getBoundingClientRect().bottom || 0, right: window.innerWidth - (langMenuRef.current?.getBoundingClientRect().right || 0) }}
                >
                  {locales.map((lng) => (
                    <button
                      key={lng}
                      onClick={() => handleLocaleChange(lng)}
                      className={`w-full px-4 py-2 text-left font-inter text-lg ${
                        locale === lng ? 'font-bold text-[#002D64]' : 'font-normal text-[#1C1C1C]'
                      } hover:bg-gray-200`}
                    >
                      {lng === 'ru' ? 'РУС' : lng === 'en' ? 'ENG' : lng === 'ar' ? 'العربية' : lng === 'zh' ? '中文' : lng.toUpperCase()}
                    </button>
                  ))}
                </div>,
                document.body
              )}
            </div>
            {/* Кнопка меню */}
            <button 
              onClick={toggleMenu}
              className="flex flex-col gap-1 cursor-pointer"
              aria-label="Toggle menu"
            >
              {isMenuOpen ? (
                // Крестик для закрытия меню
                <div className="w-[31px] h-[31px] relative">
                  <Image
                    src="/images/lake-token/header/Group 704.svg"
                    alt="Close menu"
                    fill
                    sizes="31px"
                    className="object-contain"
                  />
                </div>
              ) : (
                // Гамбургер меню
                <div className="w-[40px] h-[14px] flex flex-col justify-between">
                  <div className="w-10 h-1 bg-[#002D64]"></div>
                  <div className="w-10 h-1 bg-[#002D64]"></div>
                </div>
              )}
            </button>
          </div>
        </div>
      </div>
      {/* Выпадающее меню через Portal */}
      {typeof window !== 'undefined' && isMenuOpen && createPortal(
        <div
          className="fixed left-0 w-full bg-[#EFEFEF] z-[100000] border-t border-gray-200 transition-all duration-300 ease-out animate-slideDownFadeInMenu"
          style={{ top: '72px' }}
        >
          <div className="max-w-[1728px] mx-auto px-4 lg:px-[294px] py-8">
            <nav className="flex flex-col items-center gap-5">
              {menuItems.map((item, index) => (
                item.path ? (
                  <Link
                    key={item.key}
                    href={`/${locale}${item.path}`}
                    className={`text-lg ${item.weight} text-[#1C1C1C] hover:text-[#002D64] transition-colors duration-200`}
                    onClick={() => setIsMenuOpen(false)}
                  >
                    {t(item.key)}
                  </Link>
                ) : (
                  <a
                    key={item.key}
                    href={`#${item.key}`}
                    className={`text-lg ${item.weight} ${
                      item.key === 'lake_token' 
                        ? 'text-[#002D64]' 
                        : 'text-[#1C1C1C]'
                    } hover:text-[#002D64] transition-colors duration-200`}
                    onClick={() => setIsMenuOpen(false)}
                  >
                    {t(item.key)}
                  </a>
                )
              ))}
            </nav>
          </div>
        </div>,
        document.body
      )}
    </header>
  );
} 